<?php
/*
 * Plugin Name: ZiNiPay 
 * Plugin URI: https://wordpress.org/plugins/zinipay
 * Description: This plugin allows your customers to pay with Bkash, Nagad, Rocket, and all BD gateways via ZiNiPay.
 * Author: Abu Sayed Al Siam
 * Author URI: https://github.com/codewithsiam
 * Version: 1.0.0
 * Requires at least: 5.2
 * Requires PHP: 7.2
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: zinipay
 */

/*
 * This action hook registers our PHP class as a WooCommerce payment gateway
 */
add_action('plugins_loaded', 'zinipay_init_gateway_class');

function zinipay_init_gateway_class()
{
    if (!class_exists('WC_Payment_Gateway')) return;

    class WC_zinipay_Gateway extends WC_Payment_Gateway
    {
        public function __construct()
        {
            $this->id = 'zinipay';
            // $this->icon = 'https://i.ibb.co.com/5W25dFyM/favicon-16x16.png';
            $this->icon = 'https://i.ibb.co.com/4w1TR0kG/Pay-with-zinipay-3.png';
            $this->has_fields = false;
            $this->method_title = __('ZiNiPay', 'zinipay');
            $this->method_description = __('Pay With ZiNiPay', 'zinipay');

            $this->supports = array(
                'products',
                'woocommerce-blocks'
            );

            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->enabled = $this->get_option('enabled');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_api_' . strtolower(get_class($this)), array($this, 'handle_webhook'));
        }

        public function init_form_fields()
        {
            $this->form_fields = array(
                'enabled' => array(
                    'title'       => 'Enable/Disable',
                    'label'       => 'Enable ZiNiPay',
                    'type'        => 'checkbox',
                    'description' => '',
                    'default'     => 'no'
                ),
                'title' => array(
                    'title'       => 'Title',
                    'type'        => 'text',
                    'description' => 'This controls the title which the user sees during checkout.',
                    'default'     => 'ZiNiPay Gateway',
                    'desc_tip'    => true,
                ),
                'description' => array(
                    'title'       => 'Description',
                    'type'        => 'textarea',
                    'description' => 'This controls the description which the user sees during checkout.',
                    'default'     => 'Pay securely with Bkash, Nagad, Rocket, and all BD gateways via ZiNiPay.',
                    'desc_tip'    => true,
                ),
                'apikeys' => array(
                    'title'       => 'Enter API Key',
                    'type'        => 'text',
                    'description' => '',
                    'default'     => '###################',
                    'desc_tip'    => true,
                ),
                'currency_rate' => array(
                    'title'       => 'Enter USD Rate',
                    'type'        => 'number',
                    'description' => '',
                    'default'     => '110',
                    'desc_tip'    => true,
                ),
                'is_digital' => array(
                    'title'       => 'Enable/Disable Digital product',
                    'label'       => 'Enable Digital product',
                    'type'        => 'checkbox',
                    'description' => '',
                    'default'     => 'no'
                ),
                'payment_site' => array(
                    'title'             => 'Payment Site URL',
                    'type'              => 'text',
                    'description'       => '',
                    'default'           => 'https://api.zinipay.com/v1/',
                    'desc_tip'          => true,
                    'custom_attributes' => array(
                        'readonly' => 'readonly'
                    ),
                ),
            );
        }


        public function process_payment($order_id)
        {
            global $woocommerce;
            $order = wc_get_order($order_id);
            $current_user = wp_get_current_user();

            $subtotal = WC()->cart->subtotal;
            $shipping_total = WC()->cart->get_shipping_total();
            $fees = WC()->cart->get_fee_total();
            $discount_excl_tax_total = WC()->cart->get_cart_discount_total();
            $discount_tax_total = WC()->cart->get_cart_discount_tax_total();
            $discount_total = $discount_excl_tax_total + $discount_tax_total;
            $total = $subtotal + $shipping_total + $fees - $discount_total;

            if ($order->get_currency() == 'USD') {
                $total = $total * $this->get_option('currency_rate');
            }

            if ($order->get_status() != 'completed') {
                $order->update_status('pending', __('Customer is being redirected to ZiNiPay', 'zinipay'));
            }

            $last_name  = $current_user->user_lastname;
            $cus_email = $order->get_billing_email();

            $data = array(
                "cus_name"     => $last_name ?: "user",
                "cus_email"    => $cus_email ?: "user@example.com",
                "amount"      => $total,
                "webhook_url" => site_url('/?wc-api=wc_zinipay_gateway&order_id=' . $order->get_id()),
                "redirect_url" => $this->get_return_url($order),
                "cancel_url"  => wc_get_checkout_url()
            );

            $header = array(
                "apikey" => $this->get_option('apikeys'),
                "url" => "https://api.zinipay.com/v1/payment/create"
            );

            $response = $this->create_payment($data, $header);
            $data = json_decode($response, true);

            // Handle API Response
            if (isset($data['status']) && $data['status'] === true && isset($data['payment_url'])) {
                return array(
                    'result'   => 'success',
                    'redirect' => $data['payment_url']
                );
            } else {
                // wc_add_notice(__('Payment initialization failed. Please try again.', 'zinipay'), 'error');
                // Get error message from API response if available
                $api_error_message = isset($data['message']) ? esc_html($data['message']) . '. ' : '';

                // Combine API error message before the main message
                $error_message = __('Payment initialization failed.  Error: ', 'zinipay') . $api_error_message;

                // Show error notice in WooCommerce
                wc_add_notice($error_message, 'error');
                return array(
                    'result'   => 'failure',
                    'redirect' => wc_get_checkout_url()
                );
            }
        }

        public function create_payment($data = "", $header = '')
        {
            $headers = array(
                'Content-Type: application/json',
                'zini-api-key: ' . $header['apikey'],
            );
            $url = $header['url'];
            $curl = curl_init();
            $data = json_encode($data);

            curl_setopt_array($curl, array(
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS => $data,
                CURLOPT_HTTPHEADER => $headers,
                CURLOPT_VERBOSE => true
            ));
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);

            $response = curl_exec($curl);
            curl_close($curl);
            return $response;
        }


        public function update_order_status($order)
        {
            $valId = isset($_REQUEST['val_id']) ? $_REQUEST['val_id'] : null;
            $invoiceId = isset($_REQUEST['invoiceId']) ? $_REQUEST['invoiceId'] : null;

            if (!$valId && !$invoiceId) {
                $order->update_status('on-hold', __('ZiNiPay payment verification failed: No val_id or invoiceId provided.', 'zinipay'));
                return false;
            }

            $data = $valId ? ["val_id" => $valId] : ["invoiceId" => $invoiceId];

            $header = array(
                "apikey" => $this->get_option('apikeys'),
                "url" => "https://api.zinipay.com/v1/payment/verify"
            );

            $response = $this->create_payment($data, $header);
            $data = json_decode($response, true);


            if (!isset($data['status']) || $data['status'] !== "COMPLETED") {
                $order->update_status('on-hold', __('ZiNiPay payment verification failed. Please check manually.', 'zinipay'));
                return false;
            }

            // Extract necessary data
            $transaction_id = $data['transaction_id'] ?? 'Unknown';
            $invoice_id = $data['invoice_id'] ?? 'Unknown';
            $amount = $data['amount'] ?? 'Unknown';
            $customer_email = $data['cus_email'] ?? 'Unknown';
            $sender_number = $data['senderNumber'] ?? 'Unknown';
            $payment_provider = $data['provider'] ?? 'Unknown';
            $payment_method = 'zinipay';

            if ($order->get_status() !== 'completed') {
                if ($this->get_option('is_digital') === 'yes') {
                    $order->update_status('completed', __("ZiNiPay payment was successfully completed. Payment Method: {$payment_method}, Amount: {$amount}, Transaction ID: {$transaction_id}, Provider: {$payment_provider}, Sender: {$sender_number}, Email: {$customer_email}, Invoice Id: {$invoice_id}", 'zinipay'));
                    $order->reduce_order_stock();
                    $order->add_order_note(__('Payment completed via ZiNiPay. trx id: ' . $transaction_id, 'zinipay'));
                    $order->payment_complete();
                } else {
                    $order->update_status('processing', __("ZiNiPay payment successfully processed. Payment Method: {$payment_method}, Amount: {$amount}, Transaction ID: {$transaction_id}, Provider: {$payment_provider}, Sender: {$sender_number}, Email: {$customer_email}, Invoice Id: {$invoice_id}", 'zinipay'));
                    $order->reduce_order_stock();
                    $order->payment_complete();
                }
            }
            return true;
        }


        public function handle_webhook()
        {
            $order_id = $_GET['order_id'];

            // Log the response to WooCommerce logs
            $logger = wc_get_logger();
            $logger->info('ZiNiPay handle_webhook: ' . print_r($order_id, true), array('source' => 'zinipay'));
            $order = wc_get_order($order_id);



            if ($order) {
                $this->update_order_status($order);
            }

            status_header(200);
            echo json_encode(['message' => 'Webhook received and processed.']);
            exit();
        }
    }

    function zinipay_add_gateway_class($gateways)
    {
        $gateways[] = 'WC_zinipay_Gateway';
        return $gateways;
    }
    add_filter('woocommerce_payment_gateways', 'zinipay_add_gateway_class');
}

function zinipay_handle_webhook()
{
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        status_header(405);
        echo json_encode(['error' => 'Invalid request method']);
        exit();
    }

    // Get raw JSON data from webhook request
    $jsonData = file_get_contents("php://input");
    $requestData = json_decode($jsonData, true);

    // Extract transaction details
    $valId = isset($requestData['val_id']) ? $requestData['val_id'] : null;
    $invoiceId = isset($requestData['invoiceId']) ? $requestData['invoiceId'] : null;
    $order_id = isset($_GET['success1']) ? $_GET['success1'] : null;

    if (!$valId && !$invoiceId) {
        status_header(400);
        echo json_encode(['error' => 'Missing transaction ID']);
        exit();
    }

    if (!$order_id) {
        status_header(400);
        echo json_encode(['error' => 'Missing order ID']);
        exit();
    }

    $data = $valId ? ["val_id" => $valId] : ["invoiceId" => $invoiceId];

    $header = array(
        "api" => get_option('apikeys'),
        "url" => "https://api.zinipay.com/v1/payment/verify"
    );

    $response = create_payment($data, $header);
    $data = json_decode($response, true);


    if (!isset($data['status']) || $data['status'] !== "success") {
        status_header(400);
        echo json_encode(['error' => 'Payment verification failed']);
        exit();
    }

    // Fetch order
    $order = wc_get_order($order_id);
    if (!$order) {
        status_header(404);
        echo json_encode(['error' => 'Order not found']);
        exit();
    }

    // Extract payment details
    $transaction_id = $data['transaction_id'] ?? 'Unknown';
    $invoice_id = $data['invoice_id'] ?? 'Unknown';
    $amount = $data['amount'] ?? 'Unknown';
    $sender_number = $data['cus_email'] ?? 'Unknown';
    $payment_method = 'zinipay';

    if ($order->get_status() !== 'completed') {
        $order->update_status('completed', __("ZiNiPay payment confirmed via webhook. Payment Method: {$payment_method}, Amount: {$amount}, Transaction ID: {$transaction_id}, Sender: {$sender_number}, Invoice Id: {$invoice_id}", 'zinipay'));
        $order->reduce_order_stock();
        $order->payment_complete();
    }

    status_header(200);
    echo json_encode(['message' => 'Webhook received and processed successfully']);
    exit();
}


add_action('rest_api_init', function () {
    register_rest_route('zinipay/v1', '/webhook', array(
        'methods' => 'POST',
        'callback' => 'zinipay_handle_webhook',
    ));
});
