<?php

/**
 * @author ZiniPay <info@zinipay.com>
 *
 * @contributor Abu Sayed Al Siam <[info@zinipay.com]>
 *
 * @created 28-11-2025
 */

namespace Modules\ZiniPay\Processor;

use Illuminate\Support\Facades\Http;
use Modules\Gateway\Contracts\{
    PaymentProcessorInterface,
    RequiresCallbackInterface
};
use Modules\Gateway\Facades\GatewayHelper;
use Modules\ZiniPay\Entities\ZiniPay;
use Modules\ZiniPay\Response\ZiniPayResponse;

class ZiniPayProcessor implements PaymentProcessorInterface, RequiresCallbackInterface
{
    private $code;

    private $apiKey;

    private $baseUrl;

    private $helper;

    private $purchaseData;

    private $successUrl;

    private $cancelUrl;

    private $requestBody;
    

    public function __construct()
    {
        $this->helper = GatewayHelper::getInstance();
    }

    /**
     * Handles payment for ZiniPay
     *
     * @param \Illuminate\Http\Request
     * @return ZiniPayResponse
     */
    public function pay($request)
    {
        $this->ziniPaySetup($request);

        $charge = $this->charge();

        if (! $charge || !isset($charge->payment_url) ) {
            throw new \Exception(__('Payment Request failed due to some issues. Please try again later.'));
        }

        return redirect($charge->payment_url);
    }

    /**
     * ZiniPay data setup
     *
     * @param \Illuminate\Http\Request
     *
     * return mixed
     */
    private function ziniPaySetup($request)
    {
        try {
            
            $this->code = $this->helper->getPaymentCode();
            $this->purchaseData = $this->helper->getPurchaseData($this->code);

            $ziniPay = ZiniPay::firstWhere('alias', 'zinipay')->data;

            $this->apiKey = $ziniPay->apiKey;

            $this->baseUrl = $ziniPay->baseUrl;


            $this->successUrl = route(moduleConfig('gateway.payment_callback'), withOldQueryIntegrity(['gateway' => 'zinipay']));
            $this->cancelUrl = route(moduleConfig('gateway.payment_cancel'), withOldQueryIntegrity(['gateway' => 'zinipay']));

            $this->requestBody = $this->getRequestBody();

        } catch (\Exception $e) {
            paymentLog($e);
            throw new \Exception(__('Error while trying to setup zinipay.'));
        }
    }

    /**
     * Create charge for payment
     *
     * @return mixed
     */
    private function charge()
    {
        try {

            if (strtoupper($this->purchaseData->currency_code) != 'BDT') {
                throw new \Exception(__('Currency not supported by merchant'));
            }

            $response = Http::withHeaders([
                    'ZINIPAY-API-KEY' => $this->apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ])
                ->post($this->baseUrl . '/api/checkout-v2', $this->requestBody);

            return json_decode($response->body()); 

        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());
        }
    }


    private function getRequestBody()
    {
        return [
            'full_name'     => auth()->check() ?? auth()->name,
            'email'         => auth()->check() ?? auth()->email,
            'amount'        => number_format($this->purchaseData->total, 2),
            'metadata'      => [
                'code'   => $this->code,
            ],
            'redirect_url'  => $this->successUrl,
            'return_type'   => 'GET',
            'cancel_url'    => $this->cancelUrl,
            'webhook_url'   => $this->successUrl
        ];
    }



    public function validateTransaction($request)
    {
        try {
            $this->ziniPaySetup($request);

            if (!isset($request->invoice_id)) {
                throw new \Exception(__('Payment verification error with ZiniPay'));
            }


            $originalIntegrity = $request->input('integrity');
            $modifiedintegrity = str_replace(' ', '+', $originalIntegrity);
            $request->merge(['integrity' => $modifiedintegrity]);

            $fields = [
                'invoice_id'     => $request->invoice_id
            ];

            $response = Http::withHeaders([
                    'ZINIPAY-API-KEY' => $this->apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ])
                ->post($this->baseUrl . '/api/verify-payment', $fields);

            $ziniPayResponse =  json_decode($response->body()); 

            return new ZiniPayResponse($this->purchaseData,  $ziniPayResponse);

        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());
        }

    }

    public function cancel($request)
    {
        throw new \Exception(__('Payment cancelled from ZiniPay.'));
    }
}







